/* GPS to SD-Card Data Logger
	This project combines a GPS receiver module and a SD memory card 
	to make a GPS data logger. 
 
	GPS fix status (showing green)
	SD card led (blinking red) if errors
 
	SD Card
	* SD card attached to SPI bus as follows:
	** MISO - pin 50 ( jaune )
	** MOSI - pin 51 ( orange )
	** Serial CLK (SCK) - pin 52 ( vert )
	** Chip Select (CS) - pin 53 ( bleu )
	
	DONNEES ATTENDUES :
	GPRMC DATE HEURE LATITUDE LONGITUDE VITESSE BOUSSOLE
	GPGGA SATELLITES_FIXES NB_SATELLITES PRECISION ALTITUDE
	SDDPT PROFONDEUR
	YXMTW TEMPERATURE
	
	EXEMPLE :
	GPRMC 030613 104829.00 48.2498500 -004.2917979 0.183 NA
	GPGGA 2 10 1.19 92.0
	PRDEP X.XXX
	YXMTW XX.X
	( 55c + 20c + 11c + 10c = 106c )
	Marge :
	( 64c + 24c + 14c + 10c = 112c)
	Memoire : 112 * 62 = 7056
*/

#include <SD.h>
#include <nmea.h>
 
// Set the pins used
//======================
// GPS CHIPSET on Serial1
//--------------------
#define GPS_RX_Pin 19
#define GPS_TX_Pin 18

// SD CARD
//--------------------
#define chipSelect 53	// CS 

// DEPTH PROBE on Serial2
//--------------------


// Warning LED 
#define INFO_Pin  9
#define ERROR_Pin 13

/*
 Green LED, fix GPS
 ------------------
 1 : Data has been written on SD Card

 Red LED, errors
 ------------------
 0 : OK
 1 : SD Card not present
 3 : Cannot write file
 5 : GPS not present
 7 : GPS not readable
 9 : GPS not initialized
*/
#define OK__DATA_WRITTEN 1
#define OK__DATA_READING 1

#define ERROR__NO_ERROR 0
#define ERROR__SD_CARD_NOT_AVAILABLE 1
#define ERROR__SD_CARD_NOT_WRITABLE 3
#define ERROR__GPS_NOT_AVAILABLE 5
#define ERROR__GPS_NOT_READABLE 7
#define ERROR__GPS_NOT_INITIALIZED 9

#define ERROR__SD_CARD_NOT_AVAILABLE 4
#define ERROR__PROBE_NOT_AVAILABLE 8
#define ERROR__PROBE_NOT_READABLE 2

#define TERMINAL_TX_RATE 9600
#define GPS_RX_RATE 9600
#define SD_TX_RATE 4800

#define DEBUG 1
#define LED_INFO_PRESENT 0
#define LED_TEMPS_CLIGNEMENT 80

#define GPS_GPRMC_NO_CHAMP_HEURE 1
#define GPS_GPRMC_NO_CHAMP_DATE 9
#define GPS_GPRMC_NO_CHAMP_LATITUDE 3
#define GPS_GPRMC_NO_CHAMP_HEMISPHERE 4
#define GPS_GPRMC_NO_CHAMP_LONGITUDE 5
#define GPS_GPRMC_NO_CHAMP_MERIDIEN 6
#define GPS_GPRMC_NO_CHAMP_SYNCHRONE 2
#define GPS_GPRMC_NO_CHAMP_VITESSE 7
#define GPS_GPRMC_NO_CHAMP_BOUSSOLE 10

#define GPS_GPGGA_NO_CHAMP_FIX 6
#define GPS_GPGGA_NO_CHAMP_NB_SATELLITES 7
#define GPS_GPGGA_NO_CHAMP_PRECISION 8
#define GPS_GPGGA_NO_CHAMP_ALTITUDE 9

int sd_card_available = 0;
int sd_card_writable = 0;
int sd_card_nb_boucles = 0;
#define SD_CARD_NB_BOUCLES_MAX 9

#define BUFFER_SIZE 112
#define TRAME_LONGUEUR 112
// SD_CARD_NB_BOUCLES_MAX * TRAME_LONGUEUR
#define MEMOIRE_LONGUEUR 1008

NMEA phrase(ALL);

String buffer;
char memoire[MEMOIRE_LONGUEUR];
char buffer_str[BUFFER_SIZE];

int errorLevel=0;
char data_sondeur;
File logfile;

String gprmc = String("GPRMC");
String gprmc_valeur;

String gpgga = String("GPGGA");
String gpgga_valeur;

String sddpt = String("SDDPT");
String sddpt_valeur;

String yxmtw = String("YXMTW");
String yxmtw_valeur;


char* fichier_nom = "OSMOSE.TXT";
String journal_ligne;
int journal_ligne_longueur;

String rien = String("RIEN");
char gprmc_valeur_defaut[BUFFER_SIZE] = "GPRMC NA NA NA NA NA NA";
char gpgga_valeur_defaut[BUFFER_SIZE] = "GPGGA NA NA NA NA";
char sddpt_valeur_defaut[BUFFER_SIZE] = "SDDPT NA";
char yxmtw_valeur_defaut[BUFFER_SIZE] = "YXMTW NA";
String valeur_defaut = String("NA");

String date;
String separateur = String(" ");

// blink out an error code
void notice(uint8_t noticePin, uint8_t errno) {
	for (int i=0; i<errno; i++) {
		digitalWrite(noticePin, HIGH);delay(LED_TEMPS_CLIGNEMENT);
		digitalWrite(noticePin, LOW);delay(LED_TEMPS_CLIGNEMENT);
	}
	delay(5);
}

void error(uint8_t errno) { notice(ERROR_Pin, errno); }
void info(uint8_t errno) { notice(INFO_Pin, errno); }
 
int setup_sd() {
	// Chip select pin set for SD library 
	pinMode(chipSelect, OUTPUT);
	
	int errorLevel=0;
	// See if the card is present and can be initialized:
	Sd2Card card;
	sd_card_available = card.init(SPI_HALF_SPEED, chipSelect);
	sd_card_available = SD.begin(chipSelect);
	
	if( sd_card_available ) {
		
		logfile = SD.open(fichier_nom, FILE_WRITE);
		if(!logfile) {
			errorLevel = ERROR__SD_CARD_NOT_WRITABLE;
			if(DEBUG) Serial.println("! SD card not writable");
				sd_card_writable = 0;
		} else {
			sd_card_writable = 1;
			logfile.close();
		}
	}
	else 
	{
		errorLevel = ERROR__SD_CARD_NOT_AVAILABLE;
		if(DEBUG) Serial.println("! SD card not available");
	}
	return(errorLevel);
}

void fichier_lecture() {
	File fichier = SD.open(fichier_nom);
	if (fichier) {
		Serial.println(fichier_nom);
		// read from the file until there's nothing else in it:
		while (fichier.available()) {
			Serial.write(fichier.read());
		}
		// close the file:
		fichier.close();
	}
}

void setup()
{
	Serial.begin(TERMINAL_TX_RATE);
	Serial2.begin(SD_TX_RATE);
	int errorLevel = ERROR__NO_ERROR;
	Serial1.begin(GPS_RX_RATE);
	pinMode(INFO_Pin, OUTPUT);
	if(LED_INFO_PRESENT) digitalWrite(INFO_Pin, LOW);
	pinMode(ERROR_Pin, OUTPUT);
	digitalWrite(ERROR_Pin, LOW);
	setup_sd();
	// error(errorLevel);
	//if(DEBUG) Serial.println("Setup has finished.");
	init_valeurs();
}

void init_valeurs() {
	gprmc_valeur = String(gprmc_valeur_defaut);
	gpgga_valeur = String(gpgga_valeur_defaut);
	sddpt_valeur = String(sddpt_valeur_defaut);
	yxmtw_valeur = String(yxmtw_valeur_defaut);
}

boolean valeurs_defaut() {
	return gprmc_valeur.equals(gprmc_valeur_defaut)
	 || gpgga_valeur.equals(gpgga_valeur_defaut)
	 || yxmtw_valeur.equals(yxmtw_valeur_defaut);
	// || sddpt_valeur.equals(sddpt_valeur_defaut)
}

void check_sd_available() {
	if( ! sd_card_available && ! sd_card_writable ) {
		errorLevel = ERROR__SD_CARD_NOT_AVAILABLE;
	}
	// if( errorLevel == ERROR__SD_CARD_NOT_AVAILABLE) Serial.println("! SD CARD is not available");
	error(errorLevel);
}

void lecture_nmea_gps() {
	buffer = String("RIEN");
	errorLevel = ERROR__NO_ERROR;
	char sortie;
	int sequence = 0;
	boolean gps_available = ( Serial1.available() > 0 );
	while(gps_available && (errorLevel != ERROR__GPS_NOT_READABLE) ) {
		if (phrase.decode(Serial1.read()))
		{
			// DATATYPE
			String trame = String( phrase.term(0) );
			
			if( trame.equals(gprmc) || trame.equals( gpgga ) ) {
				buffer = trame;
				buffer+=separateur;
				if( trame.equals(gprmc) && phrase.gprmc_status() == 'A') {
					String latitude;
					String longitude;
					String boussole;
					char* str_boussole;
					char* cardinal;

					date = phrase.term(GPS_GPRMC_NO_CHAMP_DATE);
					buffer+=date+separateur+String( phrase.term(GPS_GPRMC_NO_CHAMP_HEURE ) )+separateur;
					latitude = String( phrase.term(GPS_GPRMC_NO_CHAMP_LATITUDE ) );
					cardinal = phrase.term(GPS_GPRMC_NO_CHAMP_HEMISPHERE);
					if( cardinal[0] == 'S') buffer+="-";
					buffer+=latitude+separateur;
					cardinal = phrase.term(GPS_GPRMC_NO_CHAMP_MERIDIEN);
					if( cardinal[0] == 'W') buffer+="-";
					longitude = String( phrase.term(GPS_GPRMC_NO_CHAMP_LONGITUDE ) );
					buffer+=longitude+separateur;
					// buffer+=String(phrase.gprmc_speed(KMPH));
					buffer+=String( phrase.term(GPS_GPRMC_NO_CHAMP_VITESSE) );
					buffer+=separateur;
					// buffer+=String(phrase.gprmc_course());
					str_boussole = phrase.term(GPS_GPRMC_NO_CHAMP_BOUSSOLE);
					boussole = String( str_boussole );
					if(boussole.length()==0) boussole = valeur_defaut;
					buffer+= boussole;
					gprmc_valeur = buffer;
				}
				if( trame.equals( gpgga ) ) {
					String altitude;
					buffer+=String( phrase.term(GPS_GPGGA_NO_CHAMP_FIX) );
					buffer+=separateur;
					buffer+=String( phrase.term(GPS_GPGGA_NO_CHAMP_NB_SATELLITES)) ;
					buffer+=separateur;
					buffer+=String(phrase.term(GPS_GPGGA_NO_CHAMP_PRECISION));
					buffer+=separateur;
					altitude = String(phrase.term(GPS_GPGGA_NO_CHAMP_ALTITUDE));
					if(altitude.length()==0) altitude = valeur_defaut;
					buffer+=altitude;
					gpgga_valeur = buffer;
				}
			}
			else
			{
				errorLevel = ERROR__GPS_NOT_READABLE;
			}
		} else {
			errorLevel = ERROR__GPS_NOT_AVAILABLE;
		}
		gps_available = ( Serial1.available() > 0 );
		if( !gps_available ) {
			errorLevel = ERROR__GPS_NOT_AVAILABLE;
		}
	}
	if( (errorLevel != ERROR__GPS_NOT_READABLE) && !gps_available)
	{
		errorLevel = ERROR__GPS_NOT_AVAILABLE;
	}
	if(DEBUG) {
		if( errorLevel == ERROR__GPS_NOT_INITIALIZED) Serial.println("\n! GPS is not initialized");
		//if( errorLevel == ERROR__GPS_NOT_READABLE) Serial.println("\n! GPS is not readable");
		//if( errorLevel == ERROR__GPS_NOT_AVAILABLE) Serial.println("\n! GPS is not available");
		// if( errorLevel == ERROR__SD_CARD_NOT_AVAILABLE) Serial.println("\n! SD CARD is not available");
		//error(errorLevel);
	}
}

void lecture_nmea_sonde() {
	buffer = String("RIEN");
	errorLevel = ERROR__NO_ERROR;
	boolean message_attendu = false;
	char sortie;
	int sequence = 0;
	boolean ok = false;
	boolean sonde_available = ( Serial2.available() > 0 );
	while(sonde_available && (errorLevel != ERROR__PROBE_NOT_READABLE) ) {
		if (phrase.decode(Serial2.read()))
		{
			String trame = String( phrase.term(0) );			
			if( trame.equals(sddpt) || trame.equals( yxmtw ) ) {
				buffer = trame;
				buffer+=separateur;
				if( trame.equals( sddpt ) ) {
					String profondeur = String( phrase.term(1) );
					if(profondeur.length()==0) profondeur = valeur_defaut;
					buffer+=profondeur;
					sddpt_valeur=buffer;
				}
				if( trame.equals( yxmtw ) ) {
					buffer+=String( phrase.term(1) );					
					yxmtw_valeur=buffer;
				}
			}
			else
			{
				errorLevel = ERROR__GPS_NOT_READABLE;
			}
		} else {
			errorLevel = ERROR__PROBE_NOT_AVAILABLE;
		}
		sonde_available = ( Serial2.available() > 0 );
		if( !sonde_available ) {
			errorLevel = ERROR__PROBE_NOT_AVAILABLE;
		}
	}
	if( (errorLevel != ERROR__PROBE_NOT_READABLE) && !sonde_available)
	{
		errorLevel = ERROR__PROBE_NOT_AVAILABLE;
	}
	if(DEBUG) {
		//if( errorLevel == ERROR__PROBE_NOT_INITIALIZED) Serial.println("\n! PROBE is not initialized");
		//if( errorLevel == ERROR__PROBE_NOT_READABLE) Serial.println("\n! PROBE is not readable");
		//if( errorLevel == ERROR__PROBE_NOT_AVAILABLE) Serial.println("\n! PROBE is not available");
		//error(errorLevel);
	}
}

int simul_sd_card_write() {
	int errorLevel = 0;
	Serial.println(memoire);
	info(OK__DATA_WRITTEN);
	return(errorLevel);
}

int sd_card_write() {
	int errorLevel = 0;
	if( sd_card_available ) {
		if(!logfile) logfile = SD.open(fichier_nom, FILE_WRITE);
		if(!logfile) {
			errorLevel = ERROR__SD_CARD_NOT_WRITABLE;
			if(DEBUG) Serial.println("! SD card not writable");
			sd_card_writable = 0;
		} else {
			sd_card_writable = 1;
			logfile.write(memoire);
			logfile.flush();
			logfile.close();
			info(OK__DATA_WRITTEN);
		}
	}
	return(errorLevel);
}

void loop()
{
	lecture_nmea_gps();
	lecture_nmea_sonde();

	if(errorLevel == ERROR__GPS_NOT_READABLE) {
		error(ERROR__GPS_NOT_READABLE);
		delay(100);
	}
	if(!valeurs_defaut()) {
		journal_ligne = gprmc_valeur + separateur + gpgga_valeur + separateur + sddpt_valeur + separateur + yxmtw_valeur;
		journal_ligne_longueur = journal_ligne.length();
		for(int i=0;i<journal_ligne_longueur;i++) {
			memoire[sd_card_nb_boucles*TRAME_LONGUEUR + i] = journal_ligne[i];
		}
		sd_card_nb_boucles++;
		if( sd_card_nb_boucles > SD_CARD_NB_BOUCLES_MAX - 1 ) {
			// sd_card_write();
			simul_sd_card_write();
			sd_card_nb_boucles = 0;
		} else {
			for(int i=journal_ligne_longueur;i<TRAME_LONGUEUR-2;i++) {
				memoire[(sd_card_nb_boucles-1)*TRAME_LONGUEUR + i] = ' ';
			}
			memoire[(sd_card_nb_boucles-1)*TRAME_LONGUEUR+TRAME_LONGUEUR-2] = 10;
			memoire[(sd_card_nb_boucles-1)*TRAME_LONGUEUR+TRAME_LONGUEUR-1] = 13;
		}
		init_valeurs();
	}
}
